import tkinter as tk
import math

class TwoPlayerAirHockey:
    def __init__(self, root):
        self.root = root
        self.root.title("Аэрохоккей для двоих")
        self.root.geometry("800x500")
        
        # Параметры игры
        self.canvas_width = 800
        self.canvas_height = 400
        self.puck_radius = 12
        self.mallet_radius = 20
        
        # Позиции
        self.puck_x = self.canvas_width // 2
        self.puck_y = self.canvas_height // 2
        self.player1_x = self.canvas_width // 4
        self.player1_y = self.canvas_height // 2
        self.player2_x = self.canvas_width * 3 // 4
        self.player2_y = self.canvas_height // 2
        
        # Скорости
        self.puck_vx = 0
        self.puck_vy = 0
        
        # Счет
        self.player1_score = 0
        self.player2_score = 0
        
        # Физика
        self.friction = 0.99
        
        self.setup_ui()
        self.setup_bindings()
        self.game_loop()
    
    def setup_ui(self):
        # Холст для игры
        self.canvas = tk.Canvas(self.root, width=self.canvas_width, 
                               height=self.canvas_height, bg="darkgreen")
        self.canvas.pack(fill=tk.BOTH, expand=True)
        
        # Панель счета
        score_frame = tk.Frame(self.root, bg="lightgray", height=50)
        score_frame.pack(fill=tk.X)
        score_frame.pack_propagate(False)
        
        self.score_label = tk.Label(score_frame, text="Игрок 1: 0 - 0 :Игрок 2", 
                                   font=("Arial", 16, "bold"), bg="lightgray")
        self.score_label.pack(expand=True)
        
        # Инструкция
        info_frame = tk.Frame(self.root, bg="lightyellow", height=30)
        info_frame.pack(fill=tk.X)
        info_frame.pack_propagate(False)
        
        info_text = "Игрок 1: WASD | Игрок 2: Стрелки"
        info_label = tk.Label(info_frame, text=info_text, font=("Arial", 12), bg="lightyellow")
        info_label.pack(expand=True)
        
        # Рисуем поле
        self.draw_field()
    
    def draw_field(self):
        # Очищаем canvas
        self.canvas.delete("all")
        
        # Центральная линия
        self.canvas.create_line(self.canvas_width // 2, 0, 
                               self.canvas_width // 2, self.canvas_height, 
                               fill="white", width=2)
        
        # Центральный круг
        center_x, center_y = self.canvas_width // 2, self.canvas_height // 2
        self.canvas.create_oval(center_x - 40, center_y - 40,
                               center_x + 40, center_y + 40,
                               outline="white", width=2)
        
        # Вратарские площадки
        goal_width = 80
        goal_depth = 15
        
        # Левые ворота (игрок 1 защищает)
        self.canvas.create_rectangle(0, center_y - goal_width // 2,
                                   goal_depth, center_y + goal_width // 2,
                                   outline="white", width=2)
        
        # Правые ворота (игрок 2 защищает)
        self.canvas.create_rectangle(self.canvas_width - goal_depth, center_y - goal_width // 2,
                                   self.canvas_width, center_y + goal_width // 2,
                                   outline="white", width=2)
    
    def setup_bindings(self):
        # Привязка клавиш для обоих игроков
        self.root.bind('<KeyPress>', self.key_press)
        self.root.bind('<KeyRelease>', self.key_release)
        self.root.focus_set()
        
        # Скорости клюшек
        self.player1_vx = 0
        self.player1_vy = 0
        self.player2_vx = 0
        self.player2_vy = 0
    
    def key_press(self, event):
        speed = 6
        
        # Игрок 1 (WASD)
        if event.keysym == 'w':
            self.player1_vy = -speed
        elif event.keysym == 's':
            self.player1_vy = speed
        elif event.keysym == 'a':
            self.player1_vx = -speed
        elif event.keysym == 'd':
            self.player1_vx = speed
        
        # Игрок 2 (Стрелки)
        elif event.keysym == 'Up':
            self.player2_vy = -speed
        elif event.keysym == 'Down':
            self.player2_vy = speed
        elif event.keysym == 'Left':
            self.player2_vx = -speed
        elif event.keysym == 'Right':
            self.player2_vx = speed
    
    def key_release(self, event):
        # Игрок 1
        if event.keysym in ['w', 's']:
            self.player1_vy = 0
        elif event.keysym in ['a', 'd']:
            self.player1_vx = 0
        
        # Игрок 2
        elif event.keysym in ['Up', 'Down']:
            self.player2_vy = 0
        elif event.keysym in ['Left', 'Right']:
            self.player2_vx = 0
    
    def move_players(self):
        # Двигаем игрока 1 (левая половина)
        new_x1 = self.player1_x + self.player1_vx
        new_y1 = self.player1_y + self.player1_vy
        
        # Ограничиваем движение игрока 1 его половиной
        if (self.mallet_radius <= new_x1 <= self.canvas_width // 2 - self.mallet_radius):
            self.player1_x = new_x1
        if (self.mallet_radius <= new_y1 <= self.canvas_height - self.mallet_radius):
            self.player1_y = new_y1
        
        # Двигаем игрока 2 (правая половина)
        new_x2 = self.player2_x + self.player2_vx
        new_y2 = self.player2_y + self.player2_vy
        
        # Ограничиваем движение игрока 2 его половиной
        if (self.canvas_width // 2 + self.mallet_radius <= new_x2 <= self.canvas_width - self.mallet_radius):
            self.player2_x = new_x2
        if (self.mallet_radius <= new_y2 <= self.canvas_height - self.mallet_radius):
            self.player2_y = new_y2
    
    def move_puck(self):
        # Двигаем шайбу
        self.puck_x += self.puck_vx
        self.puck_y += self.puck_vy
        
        # Применяем трение
        self.puck_vx *= self.friction
        self.puck_vy *= self.friction
        
        # Если скорость очень мала, останавливаем
        if abs(self.puck_vx) < 0.1:
            self.puck_vx = 0
        if abs(self.puck_vy) < 0.1:
            self.puck_vy = 0
    
    def check_collisions(self):
        # Столкновение с бортами
        if self.puck_x - self.puck_radius <= 0:
            self.puck_x = self.puck_radius
            self.puck_vx = -self.puck_vx * 0.9
        elif self.puck_x + self.puck_radius >= self.canvas_width:
            self.puck_x = self.canvas_width - self.puck_radius
            self.puck_vx = -self.puck_vx * 0.9
        
        if self.puck_y - self.puck_radius <= 0:
            self.puck_y = self.puck_radius
            self.puck_vy = -self.puck_vy * 0.9
        elif self.puck_y + self.puck_radius >= self.canvas_height:
            self.puck_y = self.canvas_height - self.puck_radius
            self.puck_vy = -self.puck_vy * 0.9
        
        # Столкновение с клюшкой игрока 1
        dx = self.puck_x - self.player1_x
        dy = self.puck_y - self.player1_y
        distance = math.sqrt(dx**2 + dy**2)
        
        if distance < self.puck_radius + self.mallet_radius:
            self.handle_collision(self.player1_x, self.player1_y, self.player1_vx, self.player1_vy)
        
        # Столкновение с клюшкой игрока 2
        dx = self.puck_x - self.player2_x
        dy = self.puck_y - self.player2_y
        distance = math.sqrt(dx**2 + dy**2)
        
        if distance < self.puck_radius + self.mallet_radius:
            self.handle_collision(self.player2_x, self.player2_y, self.player2_vx, self.player2_vy)
    
    def handle_collision(self, mallet_x, mallet_y, mallet_vx, mallet_vy):
        # Вектор от клюшки к шайбе
        dx = self.puck_x - mallet_x
        dy = self.puck_y - mallet_y
        distance = math.sqrt(dx**2 + dy**2)
        
        # Нормализуем вектор
        if distance > 0:
            dx /= distance
            dy /= distance
        
        # Отталкивание
        overlap = self.puck_radius + self.mallet_radius - distance
        self.puck_x += dx * overlap
        self.puck_y += dy * overlap
        
        # Передача импульса
        impulse = 1.2
        self.puck_vx = dx * impulse * 8 + mallet_vx * 0.3
        self.puck_vy = dy * impulse * 8 + mallet_vy * 0.3
    
    def check_goals(self):
        center_y = self.canvas_height // 2
        goal_width = 80
        
        # Гол в левых воротах (игрок 2 забил)
        if (self.puck_x - self.puck_radius <= 15 and 
            center_y - goal_width // 2 <= self.puck_y <= center_y + goal_width // 2):
            self.player2_score += 1
            self.reset_after_goal("Игрок 2 забил гол!")
        
        # Гол в правых воротах (игрок 1 забил)
        elif (self.puck_x + self.puck_radius >= self.canvas_width - 15 and 
              center_y - goal_width // 2 <= self.puck_y <= center_y + goal_width // 2):
            self.player1_score += 1
            self.reset_after_goal("Игрок 1 забил гол!")
    
    def reset_after_goal(self, message):
        # Показываем сообщение
        self.canvas.create_text(self.canvas_width // 2, self.canvas_height // 2,
                              text=message, font=("Arial", 20, "bold"), fill="yellow")
        self.root.update()
        
        # Ждем немного
        self.root.after(1000)
        
        # Сбрасываем позиции
        self.puck_x = self.canvas_width // 2
        self.puck_y = self.canvas_height // 2
        self.puck_vx = 0
        self.puck_vy = 0
        
        self.player1_x = self.canvas_width // 4
        self.player1_y = self.canvas_height // 2
        self.player2_x = self.canvas_width * 3 // 4
        self.player2_y = self.canvas_height // 2
        
        # Обновляем счет
        self.score_label.config(text=f"Игрок 1: {self.player1_score} - {self.player2_score} :Игрок 2")
    
    def draw_objects(self):
        # Очищаем и перерисовываем поле
        self.draw_field()
        
        # Рисуем клюшку игрока 1 (синяя)
        self.canvas.create_oval(
            self.player1_x - self.mallet_radius,
            self.player1_y - self.mallet_radius,
            self.player1_x + self.mallet_radius,
            self.player1_y + self.mallet_radius,
            fill="blue", outline="darkblue", width=2
        )
        
        # Рисуем клюшку игрока 2 (красная)
        self.canvas.create_oval(
            self.player2_x - self.mallet_radius,
            self.player2_y - self.mallet_radius,
            self.player2_x + self.mallet_radius,
            self.player2_y + self.mallet_radius,
            fill="red", outline="darkred", width=2
        )
        
        # Рисуем шайбу (черная)
        self.canvas.create_oval(
            self.puck_x - self.puck_radius,
            self.puck_y - self.puck_radius,
            self.puck_x + self.puck_radius,
            self.puck_y + self.puck_radius,
            fill="black", outline="white", width=2
        )
    
    def game_loop(self):
        # Основной игровой цикл
        self.move_players()
        self.move_puck()
        self.check_collisions()
        self.check_goals()
        self.draw_objects()
        
        # Продолжаем цикл
        self.root.after(30, self.game_loop)  # ~33 FPS

def main():
    root = tk.Tk()
    game = TwoPlayerAirHockey(root)
    root.mainloop()

if __name__ == "__main__":
    main()
