import tkinter as tk
import math
import random
'''
Сводка физических принципов
1) Законы Ньютона - основа всей динамики

2) Сохранение импульса - при столкновениях

3) Сохранение энергии - с учетом потерь на трение

4) Вращательная динамика - угловые скорости и моменты

5) Упругость материалов - коэффициенты восстановления

6) Трение - вязкое сопротивление движению
'''
class AdvancedAirHockey:
    def __init__(self, root):
        self.root = root
        self.root.title("Продвинутый Аэрохоккей")
        self.root.geometry("900x700")
        
        # Параметры игры
        self.canvas_width = 800
        self.canvas_height = 500
        self.puck_radius = 10
        self.mallet_radius = 22
        
        # Режимы игры
        self.game_mode = "player_vs_bot"  # player_vs_player, player_vs_bot, bot_vs_bot
        self.bot_difficulty = "medium"  # easy, medium, hard
        
        # Позиции
        self.reset_positions()
        
        # Физические параметры
        self.friction_air = 0.995
        self.friction_ice = 0.998
        self.elasticity = 0.85
        self.max_puck_speed = 25
        self.mallet_mass = 2.0
        self.puck_mass = 1.0
        
        # Счет
        self.player1_score = 0
        self.player2_score = 0
        
        # Статистика
        self.shots = 0
        self.hits = 0
        self.max_speed = 0
        
        # Угловая скорость шайбы
        self.puck_spin = 0
        self.puck_angular_velocity = 0
        
        self.setup_ui()
        self.setup_bindings()
        self.game_loop()
    
    def reset_positions(self):
        # Позиции
        self.puck_x = self.canvas_width // 2
        self.puck_y = self.canvas_height // 2
        self.player1_x = self.canvas_width // 4
        self.player1_y = self.canvas_height // 2
        self.player2_x = self.canvas_width * 3 // 4
        self.player2_y = self.canvas_height // 2
        
        # Скорости
        self.puck_vx = 0
        self.puck_vy = 0
        self.player1_vx = 0
        self.player1_vy = 0
        self.player2_vx = 0
        self.player2_vy = 0
        
        # Угловая скорость
        self.puck_angular_velocity = 0
        self.puck_spin = 0
    
    def setup_ui(self):
        # Основной фрейм
        main_frame = tk.Frame(self.root)
        main_frame.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)
        
        # Верхняя панель управления
        control_frame = tk.Frame(main_frame, bg="lightgray", height=60)
        control_frame.pack(fill=tk.X)
        control_frame.pack_propagate(False)
        
        # Режим игры
        mode_frame = tk.Frame(control_frame, bg="lightgray")
        mode_frame.pack(side=tk.LEFT, padx=10)
        
        tk.Label(mode_frame, text="Режим:", bg="lightgray").pack(side=tk.LEFT)
        self.mode_var = tk.StringVar(value=self.game_mode)
        mode_menu = tk.OptionMenu(mode_frame, self.mode_var, "player_vs_player", "player_vs_bot", "bot_vs_bot")
        mode_menu.config(width=12)
        mode_menu.pack(side=tk.LEFT, padx=5)
        
        # Сложность бота
        difficulty_frame = tk.Frame(control_frame, bg="lightgray")
        difficulty_frame.pack(side=tk.LEFT, padx=10)
        
        tk.Label(difficulty_frame, text="Сложность:", bg="lightgray").pack(side=tk.LEFT)
        self.difficulty_var = tk.StringVar(value=self.bot_difficulty)
        diff_menu = tk.OptionMenu(difficulty_frame, self.difficulty_var, "easy", "medium", "hard")
        diff_menu.config(width=8)
        diff_menu.pack(side=tk.LEFT, padx=5)
        
        # Кнопка сброса
        reset_btn = tk.Button(control_frame, text="Новая игра", command=self.reset_game)
        reset_btn.pack(side=tk.RIGHT, padx=10)
        
        # Панель счета
        score_frame = tk.Frame(main_frame, bg="white", height=40)
        score_frame.pack(fill=tk.X)
        score_frame.pack_propagate(False)
        
        self.score_label = tk.Label(score_frame, text="Игрок 1: 0 - 0 :Игрок 2", 
                                   font=("Arial", 16, "bold"), bg="white")
        self.score_label.pack(expand=True)
        
        # Холст для игры
        self.canvas = tk.Canvas(main_frame, width=self.canvas_width, 
                               height=self.canvas_height, bg="#1a4d1a")
        self.canvas.pack(fill=tk.BOTH, expand=True, pady=10)
        
        # Панель статистики
        stats_frame = tk.Frame(main_frame, bg="lightyellow", height=40)
        stats_frame.pack(fill=tk.X)
        stats_frame.pack_propagate(False)
        
        self.stats_label = tk.Label(stats_frame, text="Удары: 0 | Попадания: 0 | Макс. скорость: 0 м/с", 
                                   font=("Arial", 10), bg="lightyellow")
        self.stats_label.pack(expand=True)
        
        # Инструкция
        info_frame = tk.Frame(main_frame, bg="lightblue", height=30)
        info_frame.pack(fill=tk.X)
        info_frame.pack_propagate(False)
        
        info_text = "Игрок 1: WASD | Игрок 2: Стрелки | Пробел: удар"
        info_label = tk.Label(info_frame, text=info_text, font=("Arial", 11), bg="lightblue")
        info_label.pack(expand=True)
        
        # Рисуем поле
        self.draw_field()
    
    def draw_field(self):
        # Очищаем canvas
        self.canvas.delete("all")
        
        # Лед
        self.canvas.create_rectangle(10, 10, self.canvas_width-10, self.canvas_height-10,
                                   fill="#e0f0ff", outline="#a0c0ff", width=3)
        
        # Центральная линия
        self.canvas.create_line(self.canvas_width // 2, 15, 
                               self.canvas_width // 2, self.canvas_height - 15, 
                               fill="red", width=2, dash=(5, 3))
        
        # Центральный круг
        center_x, center_y = self.canvas_width // 2, self.canvas_height // 2
        self.canvas.create_oval(center_x - 50, center_y - 50,
                               center_x + 50, center_y + 50,
                               outline="red", width=2)
        
        # Вратарские площадки
        goal_width = 100
        goal_depth = 25
        
        # Левые ворота
        self.canvas.create_rectangle(10, center_y - goal_width // 2,
                                   10 + goal_depth, center_y + goal_width // 2,
                                   outline="blue", width=3)
        
        # Правые ворота
        self.canvas.create_rectangle(self.canvas_width - 10 - goal_depth, center_y - goal_width // 2,
                                   self.canvas_width - 10, center_y + goal_width // 2,
                                   outline="red", width=3)
        
        # Точки вбрасывания
        for x in [self.canvas_width//4, self.canvas_width*3//4]:
            self.canvas.create_oval(x-3, center_y-3, x+3, center_y+3, fill="blue")
    
    def setup_bindings(self):
        self.root.bind('<KeyPress>', self.key_press)
        self.root.bind('<KeyRelease>', self.key_release)
        self.root.bind('<space>', self.slap_shot)
        self.root.focus_set()
    
    def key_press(self, event):
        speed = 8
        
        # Игрок 1 (WASD)
        if event.keysym == 'w':
            self.player1_vy = -speed
        elif event.keysym == 's':
            self.player1_vy = speed
        elif event.keysym == 'a':
            self.player1_vx = -speed
        elif event.keysym == 'd':
            self.player1_vx = speed
        
        # Игрок 2 (Стрелки)
        elif event.keysym == 'Up':
            self.player2_vy = -speed
        elif event.keysym == 'Down':
            self.player2_vy = speed
        elif event.keysym == 'Left':
            self.player2_vx = -speed
        elif event.keysym == 'Right':
            self.player2_vx = speed
    
    def key_release(self, event):
        # Игрок 1
        if event.keysym in ['w', 's']:
            self.player1_vy = 0
        elif event.keysym in ['a', 'd']:
            self.player1_vx = 0
        
        # Игрок 2
        elif event.keysym in ['Up', 'Down']:
            self.player2_vy = 0
        elif event.keysym in ['Left', 'Right']:
            self.player2_vx = 0
    
    def slap_shot(self, event):
        # Сильный удар при нажатии пробела
        current_player = None
        if abs(self.player1_vx) > 0 or abs(self.player1_vy) > 0:
            current_player = 1
        elif abs(self.player2_vx) > 0 or abs(self.player2_vy) > 0:
            current_player = 2
        
        if current_player:
            self.apply_slap_shot(current_player)
    
    def apply_slap_shot(self, player):
        if player == 1:
            mallet_x, mallet_y = self.player1_x, self.player1_y
            mallet_vx, mallet_vy = self.player1_vx, self.player1_vy
        else:
            mallet_x, mallet_y = self.player2_x, self.player2_y
            mallet_vx, mallet_vy = self.player2_vx, self.player2_vy
        
        dx = self.puck_x - mallet_x
        dy = self.puck_y - mallet_y
        distance = math.sqrt(dx**2 + dy**2)
        
        if distance < self.puck_radius + self.mallet_radius + 10:
            # Сильный удар
            power = 2.0
            angle = math.atan2(dy, dx)
            
            self.puck_vx = math.cos(angle) * power * 15 + mallet_vx * 0.5
            self.puck_vy = math.sin(angle) * power * 15 + mallet_vy * 0.5
            
            # Добавляем вращение
            self.puck_angular_velocity += random.uniform(-5, 5)
            
            self.shots += 1
            self.update_stats()
    
    def move_players(self):
        # Двигаем игрока 1
        if self.game_mode in ["player_vs_player", "player_vs_bot"]:
            new_x1 = self.player1_x + self.player1_vx
            new_y1 = self.player1_y + self.player1_vy
            
            if (self.mallet_radius <= new_x1 <= self.canvas_width // 2 - self.mallet_radius):
                self.player1_x = new_x1
            if (self.mallet_radius <= new_y1 <= self.canvas_height - self.mallet_radius):
                self.player1_y = new_y1
        else:
            # Бот 1
            self.move_bot(1)
        
        # Двигаем игрока 2
        if self.game_mode == "player_vs_player":
            new_x2 = self.player2_x + self.player2_vx
            new_y2 = self.player2_y + self.player2_vy
            
            if (self.canvas_width // 2 + self.mallet_radius <= new_x2 <= self.canvas_width - self.mallet_radius):
                self.player2_x = new_x2
            if (self.mallet_radius <= new_y2 <= self.canvas_height - self.mallet_radius):
                self.player2_y = new_y2
        else:
            # Бот 2
            self.move_bot(2)
    
    def move_bot(self, bot_number):
        if bot_number == 1:
            bot_x, bot_y = self.player1_x, self.player1_y
            defense_line = self.canvas_width // 3  # Линия обороны
        else:
            bot_x, bot_y = self.player2_x, self.player2_y
            defense_line = self.canvas_width * 2 // 3  # Линия обороны
        
        # Определяем, на чьей стороне шайба
        puck_on_bot_side = (bot_number == 1 and self.puck_x < self.canvas_width // 2) or \
                          (bot_number == 2 and self.puck_x > self.canvas_width // 2)
        
        if puck_on_bot_side:
            # Шайба на стороне бота - защищаем ворота
            target_x = defense_line
            
            # Позиция для защиты ворот
            goal_y = self.canvas_height // 2
            if self.puck_y < goal_y - 20:
                target_y = goal_y - 30
            elif self.puck_y > goal_y + 20:
                target_y = goal_y + 30
            else:
                target_y = goal_y
            
            # Если шайба близко, атакуем ее
            distance_to_puck = math.sqrt((self.puck_x - bot_x)**2 + (self.puck_y - bot_y)**2)
            if distance_to_puck < 150:
                # Атакуем шайбу для отбития
                target_x = self.puck_x
                target_y = self.puck_y
                
                # Смещаемся немного, чтобы отбить в сторону
                if bot_number == 1:
                    target_x += 20  # Отбиваем вправо
                else:
                    target_x -= 20  # Отбиваем влево
        else:
            # Шайба на стороне противника - занимаем позицию
            target_x = defense_line
            target_y = self.canvas_height // 2
        
        # Ограничиваем позицию бота его половиной
        if bot_number == 1:
            target_x = max(self.mallet_radius, min(target_x, self.canvas_width // 2 - self.mallet_radius))
        else:
            target_x = max(self.canvas_width // 2 + self.mallet_radius, min(target_x, self.canvas_width - self.mallet_radius))
        
        target_y = max(self.mallet_radius, min(target_y, self.canvas_height - self.mallet_radius))
        
        # Двигаем бота к целевой позиции
        dx = target_x - bot_x
        dy = target_y - bot_y
        distance = math.sqrt(dx**2 + dy**2)
        
        if distance > 2:  # Если не на целевой позиции
            # Настройки скорости в зависимости от сложности
            if self.bot_difficulty == "easy":
                speed = 4
                accuracy = 0.7
            elif self.bot_difficulty == "medium":
                speed = 6
                accuracy = 0.85
            else:  # hard
                speed = 8
                accuracy = 0.95
            
            # Добавляем неточность в зависимости от сложности
            if random.random() > accuracy:
                dx += random.uniform(-30, 30)
                dy += random.uniform(-20, 20)
                distance = math.sqrt(dx**2 + dy**2)
            
            if distance > 0:
                dx /= distance
                dy /= distance
            
            new_x = bot_x + dx * speed
            new_y = bot_y + dy * speed
            
            # Проверяем границы
            if bot_number == 1:
                if self.mallet_radius <= new_x <= self.canvas_width // 2 - self.mallet_radius:
                    self.player1_x = new_x
                if self.mallet_radius <= new_y <= self.canvas_height - self.mallet_radius:
                    self.player1_y = new_y
            else:
                if self.canvas_width // 2 + self.mallet_radius <= new_x <= self.canvas_width - self.mallet_radius:
                    self.player2_x = new_x
                if self.mallet_radius <= new_y <= self.canvas_height - self.mallet_radius:
                    self.player2_y = new_y
    
    def move_puck(self):
        # Обновляем угловое положение
        self.puck_spin += self.puck_angular_velocity
        self.puck_spin %= 360
        
        # Двигаем шайбу с учетом трения
        self.puck_x += self.puck_vx
        self.puck_y += self.puck_vy
        
        # Комбинированное трение (воздух + лед)
        current_friction = self.friction_air * self.friction_ice
        self.puck_vx *= current_friction
        self.puck_vy *= current_friction
        
        # Замедление вращения
        self.puck_angular_velocity *= 0.99
        
        # Ограничение максимальной скорости
        speed = math.sqrt(self.puck_vx**2 + self.puck_vy**2)
        if speed > self.max_puck_speed:
            self.puck_vx = self.puck_vx / speed * self.max_puck_speed
            self.puck_vy = self.puck_vy / speed * self.max_puck_speed
        
        # Обновляем максимальную скорость
        if speed > self.max_speed:
            self.max_speed = speed
        
        # Если скорость очень мала, останавливаем
        if abs(self.puck_vx) < 0.05:
            self.puck_vx = 0
        if abs(self.puck_vy) < 0.05:
            self.puck_vy = 0
    
    def check_collisions(self):
        # Столкновение с бортами (с учетом упругости)
        if self.puck_x - self.puck_radius <= 10:
            self.puck_x = 10 + self.puck_radius
            self.puck_vx = -self.puck_vx * self.elasticity
            # Добавляем вращение от удара о борт
            self.puck_angular_velocity += self.puck_vy * 0.1
            
        elif self.puck_x + self.puck_radius >= self.canvas_width - 10:
            self.puck_x = self.canvas_width - 10 - self.puck_radius
            self.puck_vx = -self.puck_vx * self.elasticity
            self.puck_angular_velocity += self.puck_vy * 0.1
        
        if self.puck_y - self.puck_radius <= 10:
            self.puck_y = 10 + self.puck_radius
            self.puck_vy = -self.puck_vy * self.elasticity
            self.puck_angular_velocity += self.puck_vx * 0.1
            
        elif self.puck_y + self.puck_radius >= self.canvas_height - 10:
            self.puck_y = self.canvas_height - 10 - self.puck_radius
            self.puck_vy = -self.puck_vy * self.elasticity
            self.puck_angular_velocity += self.puck_vx * 0.1
        
        # Столкновения с клюшками
        self.check_mallet_collision(1)
        self.check_mallet_collision(2)
    
    def check_mallet_collision(self, player):
        if player == 1:
            mallet_x, mallet_y = self.player1_x, self.player1_y
            mallet_vx, mallet_vy = self.player1_vx, self.player1_vy
        else:
            mallet_x, mallet_y = self.player2_x, self.player2_y
            mallet_vx, mallet_vy = self.player2_vx, self.player2_vy
        
        dx = self.puck_x - mallet_x
        dy = self.puck_y - mallet_y
        distance = math.sqrt(dx**2 + dy**2)
        
        if distance < self.puck_radius + self.mallet_radius:
            self.handle_mallet_collision(mallet_x, mallet_y, mallet_vx, mallet_vy, player)
    
    def handle_mallet_collision(self, mallet_x, mallet_y, mallet_vx, mallet_vy, player):
        # Вектор от клюшки к шайбе
        dx = self.puck_x - mallet_x
        dy = self.puck_y - mallet_y
        distance = math.sqrt(dx**2 + dy**2)
        
        if distance == 0:
            return
            
        # Нормализуем вектор
        dx /= distance
        dy /= distance
        
        # Отталкивание
        overlap = self.puck_radius + self.mallet_radius - distance
        self.puck_x += dx * overlap
        self.puck_y += dy * overlap
        
        # Расчет относительной скорости
        relative_vx = self.puck_vx - mallet_vx
        relative_vy = self.puck_vy - mallet_vy
        
        # Скалярное произведение для проекции скорости на нормаль
        dot_product = relative_vx * dx + relative_vy * dy
        
        # Упругое столкновение с учетом масс
        impulse = 2 * dot_product / (self.puck_mass + self.mallet_mass)
        
        # Новые скорости
        self.puck_vx -= impulse * self.mallet_mass * dx * 1.2
        self.puck_vy -= impulse * self.mallet_mass * dy * 1.2
        
        # Для ботов добавляем стратегическое отбитие
        if (player == 1 and self.game_mode != "player_vs_player" and self.game_mode != "player_vs_bot") or \
           (player == 2 and self.game_mode == "player_vs_bot"):
            # Бот отбивает шайбу в сторону центра или в сторону ворот противника
            if player == 1:
                # Бот 1 отбивает вправо (к воротам противника)
                self.puck_vx += abs(self.puck_vx) * 0.3
            else:
                # Бот 2 отбивает влево (к воротам противника)
                self.puck_vx -= abs(self.puck_vx) * 0.3
        
        # Добавляем вращение в зависимости от угла удара
        tangent_x = -dy
        tangent_y = dx
        tangent_speed = relative_vx * tangent_x + relative_vy * tangent_y
        self.puck_angular_velocity += tangent_speed * 0.5
        
        self.hits += 1
        self.update_stats()
    
    def check_goals(self):
        center_y = self.canvas_height // 2
        goal_width = 100
        
        # Гол в левых воротах
        if (self.puck_x - self.puck_radius <= 35 and 
            center_y - goal_width // 2 <= self.puck_y <= center_y + goal_width // 2):
            self.player2_score += 1
            self.reset_after_goal("Игрок 2 забил гол!")
        
        # Гол в правых воротах
        elif (self.puck_x + self.puck_radius >= self.canvas_width - 35 and 
              center_y - goal_width // 2 <= self.puck_y <= center_y + goal_width // 2):
            self.player1_score += 1
            self.reset_after_goal("Игрок 1 забил гол!")
    
    def reset_after_goal(self, message):
        # Показываем сообщение
        self.canvas.create_text(self.canvas_width // 2, self.canvas_height // 2,
                              text=message, font=("Arial", 24, "bold"), fill="darkred")
        self.root.update()
        
        # Ждем 1.5 секунды
        self.root.after(1500)
        
        # Сбрасываем позиции
        self.reset_positions()
        
        # Обновляем счет
        self.score_label.config(text=f"Игрок 1: {self.player1_score} - {self.player2_score} :Игрок 2")
    
    def update_stats(self):
        self.stats_label.config(text=f"Удары: {self.shots} | Попадания: {self.hits} | Макс. скорость: {self.max_speed:.1f} м/с")
    
    def reset_game(self):
        self.player1_score = 0
        self.player2_score = 0
        self.shots = 0
        self.hits = 0
        self.max_speed = 0
        self.game_mode = self.mode_var.get()
        self.bot_difficulty = self.difficulty_var.get()
        self.reset_positions()
        self.score_label.config(text="Игрок 1: 0 - 0 :Игрок 2")
        self.update_stats()
    
    def draw_objects(self):
        # Очищаем и перерисовываем поле
        self.draw_field()
        
        # Рисуем клюшку игрока 1 (синяя с узором)
        self.draw_mallet(self.player1_x, self.player1_y, "blue", 1)
        
        # Рисуем клюшку игрока 2 (красная с узором)
        self.draw_mallet(self.player2_x, self.player2_y, "red", 2)
        
        # Рисуем шайбу с вращением
        self.draw_puck()
        
        # Рисуем вектор скорости шайбы
        speed = math.sqrt(self.puck_vx**2 + self.puck_vy**2)
        if speed > 1:
            self.canvas.create_line(
                self.puck_x, self.puck_y,
                self.puck_x + self.puck_vx * 2,
                self.puck_y + self.puck_vy * 2,
                fill="yellow", width=2, arrow=tk.LAST
            )
    
    def draw_mallet(self, x, y, color, player):
        # Основной круг
        self.canvas.create_oval(
            x - self.mallet_radius, y - self.mallet_radius,
            x + self.mallet_radius, y + self.mallet_radius,
            fill=color, outline="black", width=2
        )
        
        # Внутренний круг
        inner_radius = self.mallet_radius * 0.6
        self.canvas.create_oval(
            x - inner_radius, y - inner_radius,
            x + inner_radius, y + inner_radius,
            fill="white", outline="black", width=1
        )
        
        # Полоса игрока
        stripe_color = "yellow" if player == 1 else "white"
        self.canvas.create_rectangle(
            x - self.mallet_radius, y - 3,
            x + self.mallet_radius, y + 3,
            fill=stripe_color, outline=""
        )
    
    def draw_puck(self):
        # Основной диск
        self.canvas.create_oval(
            self.puck_x - self.puck_radius, self.puck_y - self.puck_radius,
            self.puck_x + self.puck_radius, self.puck_y + self.puck_radius,
            fill="black", outline="white", width=2
        )
        
        # Вращающиеся метки
        marker_angle = math.radians(self.puck_spin)
        marker_x = self.puck_x + math.cos(marker_angle) * (self.puck_radius * 0.7)
        marker_y = self.puck_y + math.sin(marker_angle) * (self.puck_radius * 0.7)
        
        self.canvas.create_oval(
            marker_x - 2, marker_y - 2,
            marker_x + 2, marker_y + 2,
            fill="red", outline=""
        )
        
        # Вторая метка (противоположная)
        marker_x2 = self.puck_x + math.cos(marker_angle + math.pi) * (self.puck_radius * 0.7)
        marker_y2 = self.puck_y + math.sin(marker_angle + math.pi) * (self.puck_radius * 0.7)
        
        self.canvas.create_oval(
            marker_x2 - 2, marker_y2 - 2,
            marker_x2 + 2, marker_y2 + 2,
            fill="red", outline=""
        )
    
    def game_loop(self):
        # Основной игровой цикл
        self.move_players()
        self.move_puck()
        self.check_collisions()
        self.check_goals()
        self.draw_objects()
        
        # Продолжаем цикл
        self.root.after(30, self.game_loop)

def main():
    root = tk.Tk()
    game = AdvancedAirHockey(root)
    root.mainloop()

if __name__ == "__main__":
    main()
