import tkinter as tk
import random

class PlatformerGame:
    def __init__(self, root):
        self.root = root
        self.root.title("Платформер: Путь к финишу")
        self.root.geometry("800x600")
        
        # Игровые константы
        self.canvas_width = 800
        self.canvas_height = 500
        self.gravity = 0.8
        self.jump_strength = -15
        self.player_speed = 5
        
        # Состояние игры
        self.current_level = 1
        self.max_level = 3
        self.game_state = "playing"  # playing, win, game_over
        
        # Позиции игрока (теперь круг)
        self.player_x = 50
        self.player_y = 400
        self.player_radius = 15
        self.player_vx = 0
        self.player_vy = 0
        self.on_ground = False
        
        # Платформы для каждого уровня
        self.platforms = []
        self.finish_flag = None
        
        # Создаем уровни
        self.create_levels()
        self.load_level(self.current_level)
        
        self.setup_ui()
        self.setup_bindings()
        self.game_loop()
    
    def create_levels(self):
        """Создаем данные для всех уровней"""
        self.levels_data = {
            1: {
                "platforms": [
                    # (x, y, width, height)
                    (0, 450, 200, 20),      # Стартовая платформа
                    (250, 400, 100, 20),    # Первая платформа
                    (400, 350, 100, 20),    # Вторая платформа
                    (550, 300, 100, 20),    # Третья платформа
                    (700, 250, 100, 20),    # Высокая платформа
                ],
                "finish": (750, 230)  # Флажок на высокой платформе
            },
            2: {
                "platforms": [
                    (0, 450, 150, 20),
                    (200, 380, 80, 20),
                    (320, 320, 80, 20),
                    (450, 380, 80, 20),
                    (580, 300, 80, 20),
                    (700, 250, 80, 20),  # Высокая платформа для флажка
                ],
                "finish": (730, 230)
            },
            3: {
                "platforms": [
                    (0, 450, 120, 20),
                    (150, 350, 70, 20),
                    (250, 420, 70, 20),
                    (350, 320, 70, 20),
                    (450, 400, 70, 20),
                    (550, 280, 70, 20),
                    (650, 200, 70, 20),  # Самая высокая платформа
                ],
                "finish": (670, 180)
            }
        }
    
    def load_level(self, level):
        """Загружаем указанный уровень"""
        self.current_level = level
        level_data = self.levels_data[level]
        self.platforms = level_data["platforms"]
        self.finish_flag = level_data["finish"]
        
        # Сбрасываем позицию игрока
        self.player_x = 50
        self.player_y = 400 - self.player_radius
        self.player_vx = 0
        self.player_vy = 0
        self.game_state = "playing"
    
    def setup_ui(self):
        """Создаем интерфейс"""
        # Основной фрейм
        main_frame = tk.Frame(self.root)
        main_frame.pack(fill=tk.BOTH, expand=True)
        
        # Панель информации
        info_frame = tk.Frame(main_frame, bg="lightblue", height=40)
        info_frame.pack(fill=tk.X)
        info_frame.pack_propagate(False)
        
        self.info_label = tk.Label(info_frame, text="Уровень 1 - Используйте стрелки для движения и пробел для прыжка", 
                                  font=("Arial", 12), bg="lightblue")
        self.info_label.pack(expand=True)
        
        # Холст для игры
        self.canvas = tk.Canvas(main_frame, width=self.canvas_width, 
                               height=self.canvas_height, bg="lightblue")
        self.canvas.pack(fill=tk.BOTH, expand=True)
        
        # Панель управления
        control_frame = tk.Frame(main_frame, bg="lightgray", height=40)
        control_frame.pack(fill=tk.X)
        control_frame.pack_propagate(False)
        
        # Кнопки управления
        btn_frame = tk.Frame(control_frame, bg="lightgray")
        btn_frame.pack(expand=True)
        
        tk.Button(btn_frame, text="Рестарт уровня", command=self.restart_level).pack(side=tk.LEFT, padx=5)
        tk.Button(btn_frame, text="Следующий уровень", command=self.next_level).pack(side=tk.LEFT, padx=5)
        tk.Button(btn_frame, text="Предыдущий уровень", command=self.previous_level).pack(side=tk.LEFT, padx=5)
        
        # Отрисовываем начальное состояние
        self.draw_game()
    
    def setup_bindings(self):
        """Настраиваем управление"""
        self.root.bind('<KeyPress>', self.key_press)
        self.root.bind('<KeyRelease>', self.key_release)
        self.root.focus_set()
    
    def key_press(self, event):
        """Обработка нажатия клавиш"""
        if self.game_state != "playing":
            return
            
        if event.keysym == 'Left':
            self.player_vx = -self.player_speed
        elif event.keysym == 'Right':
            self.player_vx = self.player_speed
        elif event.keysym == 'space' and self.on_ground:
            self.player_vy = self.jump_strength
            self.on_ground = False
    
    def key_release(self, event):
        """Обработка отпускания клавиш"""
        if event.keysym in ['Left', 'Right']:
            self.player_vx = 0
    
    def restart_level(self):
        """Перезапуск текущего уровня"""
        self.load_level(self.current_level)
    
    def next_level(self):
        """Переход на следующий уровень"""
        if self.current_level < self.max_level:
            self.load_level(self.current_level + 1)
    
    def previous_level(self):
        """Переход на предыдущий уровень"""
        if self.current_level > 1:
            self.load_level(self.current_level - 1)
    
    def update_physics(self):
        """Обновление физики игры"""
        if self.game_state != "playing":
            return
        
        # Применяем гравитацию
        self.player_vy += self.gravity
        
        # Обновляем позицию по X
        new_x = self.player_x + self.player_vx
        
        # Проверяем границы экрана по X
        if self.player_radius <= new_x <= self.canvas_width - self.player_radius:
            self.player_x = new_x
        
        # Обновляем позицию по Y
        new_y = self.player_y + self.player_vy
        
        # Проверяем столкновения с платформами
        self.on_ground = False
        
        for platform in self.platforms:
            px, py, pwidth, pheight = platform
            
            # Проверяем столкновение круга с прямоугольником
            closest_x = max(px, min(new_x, px + pwidth))
            closest_y = max(py, min(new_y, py + pheight))
            
            distance_x = new_x - closest_x
            distance_y = new_y - closest_y
            distance_squared = distance_x**2 + distance_y**2
            
            if distance_squared < self.player_radius**2:
                # Столкновение обнаружено
                if abs(distance_y) > abs(distance_x):
                    # Вертикальное столкновение
                    if distance_y > 0:
                        # Столкновение сверху с платформой
                        new_y = closest_y + self.player_radius
                        self.player_vy = 0
                    else:
                        # Столкновение снизу с платформой
                        new_y = closest_y - self.player_radius
                        self.player_vy = 0
                        self.on_ground = True
                else:
                    # Горизонтальное столкновение
                    if distance_x > 0:
                        new_x = closest_x + self.player_radius
                    else:
                        new_x = closest_x - self.player_radius
        
        # Проверяем границы экрана по Y
        if new_y < self.player_radius:
            new_y = self.player_radius
            self.player_vy = 0
        elif new_y > self.canvas_height - self.player_radius:
            new_y = self.canvas_height - self.player_radius
            self.player_vy = 0
            self.on_ground = True
        
        self.player_y = new_y
        
        # Проверяем достижение финиша
        finish_x, finish_y = self.finish_flag
        distance_to_finish = ((self.player_x - finish_x)**2 + (self.player_y - finish_y)**2)**0.5
        
        if distance_to_finish < self.player_radius + 10:  # 10 - радиус флажка
            if self.current_level < self.max_level:
                self.game_state = "win"
            else:
                self.game_state = "completed"
    
    def draw_game(self):
        """Отрисовка игрового мира"""
        self.canvas.delete("all")
        
        # Рисуем фон
        self.canvas.create_rectangle(0, 0, self.canvas_width, self.canvas_height, fill="lightblue", outline="")
        
        # Рисуем платформы
        for platform in self.platforms:
            x, y, width, height = platform
            self.canvas.create_rectangle(x, y, x + width, y + height, fill="green", outline="darkgreen", width=2)
        
        # Рисуем флажок финиша
        self.draw_finish_flag()
        
        # Рисуем игрока (круг)
        self.draw_player()
        
        # Рисуем информацию об уровне
        self.canvas.create_text(70, 20, text=f"Уровень: {self.current_level}/{self.max_level}", 
                              fill="black", font=("Arial", 12, "bold"))
        
        # Рисуем сообщения о состоянии игры
        if self.game_state == "win":
            self.canvas.create_text(self.canvas_width//2, self.canvas_height//2,
                                  text="Уровень пройден! Нажмите 'Следующий уровень'", 
                                  fill="darkgreen", font=("Arial", 16, "bold"))
        elif self.game_state == "completed":
            self.canvas.create_text(self.canvas_width//2, self.canvas_height//2,
                                  text="Поздравляем! Вы прошли все уровни!", 
                                  fill="darkgreen", font=("Arial", 18, "bold"))
        
        # Обновляем информацию
        self.info_label.config(text=f"Уровень {self.current_level} - Используйте стрелки для движения и пробел для прыжка")
    
    def draw_player(self):
        """Отрисовка игрока (круг)"""
        # Основной круг
        self.canvas.create_oval(self.player_x - self.player_radius, self.player_y - self.player_radius,
                               self.player_x + self.player_radius, self.player_y + self.player_radius,
                               fill="red", outline="darkred", width=2)
        
        # Глаза для ориентира
        eye_radius = 3
        direction = 1 if self.player_vx >= 0 else -1
        self.canvas.create_oval(self.player_x + direction * 4 - eye_radius, self.player_y - 3 - eye_radius,
                               self.player_x + direction * 4 + eye_radius, self.player_y - 3 + eye_radius,
                               fill="white")
        self.canvas.create_oval(self.player_x + direction * 4 - eye_radius//2, self.player_y - 3 - eye_radius//2,
                               self.player_x + direction * 4 + eye_radius//2, self.player_y - 3 + eye_radius//2,
                               fill="black")
    
    def draw_finish_flag(self):
        """Отрисовка флажка финиша"""
        flag_x, flag_y = self.finish_flag
        
        # Флагшток
        self.canvas.create_line(flag_x, flag_y, flag_x, flag_y - 30, fill="brown", width=3)
        
        # Флажок
        flag_points = [flag_x, flag_y - 30, flag_x + 20, flag_y - 20, flag_x, flag_y - 10]
        self.canvas.create_polygon(flag_points, fill="red", outline="darkred")
        
        # Подпись
        self.canvas.create_text(flag_x + 10, flag_y - 35, text="FIN", fill="white", font=("Arial", 8, "bold"))
    
    def game_loop(self):
        """Основной игровой цикл"""
        self.update_physics()
        self.draw_game()
        
        # Продолжаем цикл
        self.root.after(30, self.game_loop)

def main():
    root = tk.Tk()
    game = PlatformerGame(root)
    root.mainloop()

if __name__ == "__main__":
    main()
